#include <errno.h>
#include <stdio.h>
#include <string.h>
#include <stdlib.h>
#include <time.h>
#include <unistd.h>
#include <stdbool.h>
#include <fcntl.h>

#include "sdfe.h"
#define DEBUG 1
#define SM4_BLOCK_SIZE 16 // sm4 block size 128bit

#define CRYPTO_SUCCESS 0
#define CRYPTO_FAILURE 1

typedef void *SGD_HANDLE;

struct s_context {
    void* hSessionHandle;
};


#define warning_print() \
	do { if (DEBUG) fprintf(stderr, "%s:%d:%s():\n",__FILE__, __LINE__, __FUNCTION__); } while (0)

#define error_print() \
	do { if (DEBUG) fprintf(stderr, "%s:%d:%s():\n",__FILE__, __LINE__, __FUNCTION__); } while (0)

#define error_print_msg(fmt, ...) \
	do { if (DEBUG) fprintf(stderr, "%s:%d:%s(): " fmt, __FILE__, __LINE__, __FUNCTION__, __VA_ARGS__); } while (0)

#define error_puts(str) \
            do { if (DEBUG) fprintf(stderr, "%s: %d: %s: %s\n", __FILE__, __LINE__, __FUNCTION__, str); } while (0)

static void hexdump(char* header, unsigned char* input, size_t len)
{
    unsigned long i;
    if (header)
        printf("%s: ", header);
    for (i = 0; i < len; i++) {
        printf("%02x ", input[i]);
    }
    printf("\n");
}

static int test_sm2(struct s_context* ctx)
{
	int result;

	char *PIN = "12345678";
	uint8_t in[32];
	uint8_t decout[32];

	size_t PINlen, inlen;
	unsigned int decout_len;
	int i;

	int key_index = 3;

        ECCrefPublicKey PubKey;
        ECCrefPrivateKey PriKey;

	inlen = sizeof(in);
	for (i = 0; i < inlen; i++) {
		in[i] = (uint8_t)i;
	}

	ECCCipher *cipherout = NULL;
	PINlen = strlen(PIN);

	cipherout = (ECCCipher *)malloc(sizeof(ECCCipher) + inlen - 1);
	if (!cipherout) {
		printf("OOM\n");
		return -1;
	}

//#define GEN_KEY 1
#ifdef GEN_KEY
	result = SDF_GenerateKeyPair_ECC(ctx->hSessionHandle, SGD_SM2, 256, &PubKey, &PriKey);
	if (result != SDR_OK) {
		error_print();
		return -1;
	}

	result = SDFE_ImportKeyPair_ECC(ctx->hSessionHandle, key_index, SDFE_UEK_SM2, (uint8_t *)PIN, PINlen, &PubKey, &PriKey);
	if (result != SDR_OK) {
		error_print();
		return -1;
	}

	result = SDF_GenerateKeyPair_ECC(ctx->hSessionHandle, SGD_SM2, 256, &PubKey, &PriKey);
	if (result != SDR_OK) {
		error_print();
		return -1;
	}

	result = SDFE_ImportKeyPair_ECC(ctx->hSessionHandle, key_index, SDFE_USK_SM2, (uint8_t *)PIN, PINlen, &PubKey, &PriKey);
	if (result != SDR_OK) {
		error_print();
		return -1;
	}
#endif

	result = SDF_GetPrivateKeyAccessRight(ctx->hSessionHandle, key_index, (uint8_t *)PIN, PINlen);
	if (result != SDR_OK) {
		error_print();
		goto err;
	}
	printf(" SDF_GetPrivateKeyAccessRight OK \n");

      
	printf(" ===================================================================== \n");
	printf(" 1. Test with Encryption Key \n");
	printf(" ===================================================================== \n");
	result = SDF_InternalEncrypt_ECC(ctx->hSessionHandle, key_index, SGD_SM2, in, inlen, cipherout);
	if (result != SDR_OK) {
		error_print();
		goto err;
	}

	hexdump("DATA_IN", (unsigned char *)in, inlen);
	hexdump("ENC_OUT", (unsigned char *)cipherout->C, cipherout->L);
        
        memset(decout, 0, sizeof(decout));
	result = SDF_InternalDecrypt_ECC(ctx->hSessionHandle, key_index, SGD_SM2, cipherout, decout, &decout_len);
	if (result != SDR_OK || memcmp(in, decout, decout_len) != 0) {
		error_print();
		return -1;
	}
	hexdump("DEC_OUT", (unsigned char *)decout, inlen);

	if (result != SDR_OK || memcmp(in, decout, decout_len) != 0) {
		error_print();
		goto err;
	}

	printf(" Test with Encryption Key (#%d) OK \n", key_index);
    

	printf(" ===================================================================== \n");
	printf(" 2. Test Encryption with Sign Key \n");
	printf(" ===================================================================== \n");
	result = SDF_InternalEncrypt_ECC(ctx->hSessionHandle, ECC_USE_ISK_LEVEL + key_index, SGD_SM2, in, inlen, cipherout);
	if (result != SDR_OK) {
		error_print();
		goto err;
	}

	hexdump("DATA_IN", (unsigned char *)in, inlen);
	hexdump("ENC_OUT", (unsigned char *)cipherout->C, cipherout->L);

       //verify
       result = SDFE_ExportKeyPair_ECC(ctx->hSessionHandle, key_index, SDFE_USK_SM2, &PubKey, &PriKey);
       if (result != SDR_OK) {
             return result;
       }

        memset(decout, 0, sizeof(decout));
       result = SDF_ExternalDecrypt_ECC(ctx->hSessionHandle, SGD_SM2, &PriKey, cipherout, decout, &decout_len);
       if (result != SDR_OK) {
             return result;
       }
	hexdump("DEC_OUT", (unsigned char *)decout, inlen);

	if (result != SDR_OK || memcmp(in, decout, decout_len) != 0) {
		error_print();
		goto err;
	}
	printf(" Test Encryption with Sign Key (#%d) OK \n", key_index);

	printf(" ===================================================================== \n");
	printf(" 3. Test Decryption with Sign Key \n");
	printf(" ===================================================================== \n");
       result = SDFE_ExportKeyPair_ECC(ctx->hSessionHandle, key_index, SDFE_USK_SM2, &PubKey, &PriKey);
       if (result != SDR_OK) {
             return result;
       }

       result = SDF_ExternalEncrypt_ECC(ctx->hSessionHandle, SGD_SM2, &PubKey, in, inlen, cipherout);
       if (result != SDR_OK) {
             return result;
       }


	hexdump("DATA_IN", (unsigned char *)in, inlen);
	hexdump("ENC_OUT", (unsigned char *)cipherout->C, cipherout->L);

        memset(decout, 0, sizeof(decout));
	result = SDF_InternalDecrypt_ECC(ctx->hSessionHandle, ECC_USE_ISK_LEVEL + key_index, SGD_SM2, cipherout, decout, &decout_len);
	if (result != SDR_OK || memcmp(in, decout, decout_len) != 0) {
		error_print();
		return -1;
	}
	hexdump("DEC_OUT", (unsigned char *)decout, inlen);

	if (result != SDR_OK || memcmp(in, decout, decout_len) != 0) {
		error_print();
		goto err;
	}
	printf(" Test Decryption with Sign Key (#%d) OK \n", key_index);

	printf(" ===================================================================== \n");
	printf("%s() ok\n", __FUNCTION__);
	return 1;
err:
	if(cipherout)
		free(cipherout);
	return -1;
}


/**
 * @ SM4
 *
 * @return int
 */
int main(int argc, char* argv[])
{
	struct s_context* ctx = NULL;
	int result;
	SGD_HANDLE hDeviceHandle;
	SGD_HANDLE hSessionHandle;

	//打开设备
	result = SDF_OpenDevice(&hDeviceHandle);
	if (result != SDR_OK) {
		printf("SDF_OpenDevice failed\n");
		goto err;
	}
	//创建会话
	result = SDF_OpenSession(hDeviceHandle,&hSessionHandle);
	if(result != SDR_OK){
		printf("SDF_OpenSession failed\n");
		return CRYPTO_FAILURE;
	}

	ctx = malloc(sizeof(struct s_context));
	if (!ctx) {
		printf("OOM\n");
		goto err;
	}

	ctx->hSessionHandle =hSessionHandle;

	//SM4加密算法验证
        test_sm2(ctx);
err:
	if(hSessionHandle)
		SDF_CloseSession(hSessionHandle);
	if(hDeviceHandle)
		SDF_CloseDevice(hDeviceHandle);
	if(ctx)
		free(ctx);


	return CRYPTO_SUCCESS;
}
